program talkto;

//
// --------------------------------------------
// talkto - a network communicator using netcat
//
// usage: talkto <host> <port>
// --------------------------------------------
//
// written by Robert Rozee, 2-june-2020
//
// attempts to provide a level of VT102 terminal emulation not available with
// netcat on its own. this is achieved by setting the console to RAW mode,
// mapping f1-f4, home and end keys to corrected escape sequences, and making
// the backspace key generate ^H instead of 0x7F.
//
// all control keys can be sent to the host without breaking the connection,
// exit is mapped to shift-f1, while shift-f2 places talkto to sleep in the
// background so that the terminal functions can be used. bring back to the
// foreground with 'fg'.
//
// we don't bother much with any error information from netcat, either it
// works or it does not work. diagnose any network connection problems using
// netcat on its own - just pass it the same parameters as you passed to
// talkto.
//

{$mode objfpc}{$H+}

uses cthreads, Classes, Process, Math, termIO, sysutils, BaseUnix;


const head:byte=0;
      tail:byte=0;

var keys:array[0..255] of char;
    time:array[0..255] of int64;

type TCheckThread = class(TThread)
     private
     protected
       procedure Execute; override;
     end;

// separate thread used to check stdin for input
procedure TCheckThread.Execute;
var ch:char;
begin
  repeat
    read(ch);
    keys[head]:=ch;
    time[head]:=GetTickCount64;                                // timestamp each character
    inc(head)
  until false
end;



procedure translate(S1, S2:string; var line:string);           // translate xterm keyboard escape sequence to VT220
var I:integer;
begin
  I:=pos(S1, line);
  while I<>0 do
  begin
    delete(line, I, length(S1));
    insert(S2, line, I);
    I:=pos(S1, line)
  end
end;



var tios, SAVEDtios:termios;
           AProcess:TProcess;
             buffer:string[255];
                 ch:char;
                  I:integer;
                  S:string;

begin
  if paramcount<>2 then
  begin
//  for I:=30 to 37 do write(I, #27'[0;1;',I,'m',' test text ',#27'[0m', #13#10);
    write(#13#10);
    write(' -------------------------------------------- ', #13#10);
    write(#27'[1;33m',
          ' talkto', #27'[0m',
                 ' - a network communicator using ',#27'[1;32m',
                                                 'netcat ', #27'[0m', #13#10#10);
    write(' usage: ', #27'[1;36m',
                  'talkto <host> <port>', #27'[0m', #13#10);
    write(' -------------------------------------------- ', #13#10#10);
    halt(-1)
  end;

  if not FileExists('/bin/netcat') then
  begin
    write(#13#10);
    write(' -------------------------------------------- ', #13#10);
    write(#27'[1;33m',
          ' talkto', #27'[0m',
                 ' - a network communicator using ',#27'[1;32m',
                                                 'netcat ', #27'[0m', #13#10#10);
    write(' could not locate: ', #27'[1;36m',
                           '/bin/netcat', #27'[0m', #13#10);
    write(' -------------------------------------------- ', #13#10#10);
    halt(-1)
  end;

  write(#13#10);
  write(' -------------------------------------------- ', #13#10);
  write(#27'[1;33m',
        ' talkto', #27'[0m',
               ' - a network communicator using ',#27'[1;32m',
                                               'netcat ', #27'[0m', #13#10#10);
  write('   host = ',paramstr(1), #13#10);
  write('   port = ',paramstr(2), #13#10#10);
  write(' ', #27'[7m',
         'press shift-f1 to exit, or shift-f2 to sleep', #27'[0m', #13#10);
  write(' -------------------------------------------- ', #13#10#10);

  TCheckThread.Create(false);

  TCGetAttr(0, tios);
  SAVEDtios:=tios;
  CFMakeRaw(tios);
  TCSetAttr(0, TCSANOW, tios);                 // place console in RAW mode

  try
    AProcess:=TProcess.Create(nil);
    AProcess.Executable := '/bin/netcat';
    AProcess.Parameters.Add(paramstr(1));
    AProcess.Parameters.Add(paramstr(2));
    AProcess.Options := [poUsePipes];
    AProcess.Execute;
    sleep(100);

    repeat
      if not AProcess.Running then break;

      I:=AProcess.Output.NumBytesAvailable;
      if I<>0 then                             // read from netcat, writing it to the terminal screen
      begin
        SetLength(buffer,AProcess.Output.Read(buffer[1], min(I, 255)));
        write(buffer)
      end;

      S:='';
      while head<>tail do                      // take user keyboard input, and send it on to netcat
      begin
        ch:=keys[tail];
        if ch=#127 then ch:=#8;                // translate delete character to backspace
        if (ch=#27) and ((GetTickCount64-time[tail])<50) then break;
                                               // defer processing if less than 50mS since ESC received
        inc(tail);
        S:=S+ch
      end;

      if pos(#27+'[1;2P',S)<>0 then break;     // shift-f1 -> exit
      if pos(#27+'[1;2Q',S)<>0 then            // shift-f2 -> sleep
      begin
        TCSetAttr(0, TCSANOW, SAVEDtios);
        fpkill(GetProcessID, SIGSTOP);         // go to sleep...
        sleep(100);                            // ... awake here
        TCSetAttr(0, TCSANOW, tios)
      end;

      translate(#27+'OP', #27+'[11~', S);      // f1
      translate(#27+'OQ', #27+'[12~', S);      // f2
      translate(#27+'OR', #27+'[13~', S);      // f3
      translate(#27+'OS', #27+'[14~', S);      // f4
      translate(#27+'[H', #27+'[1~', S);       // home
      translate(#27+'[F', #27+'[4~', S);       // end
      translate(#27+'[1;2P', '', S);           // shift-f1  (ignore - but we never get here)
      translate(#27+'[1;2Q', '', S);           // shift-f2  (ignore - we have just woken up)
      translate(#27+'[1;2R', #27+'[25~', S);   // shift-f3
      translate(#27+'[1;2S', #27+'[26~', S);   // shift-f4

      if length(S)<>0 then AProcess.Input.Write(S[1], length(S))
    until false
  except write(#10#13, '!! exception 1', #13#10) end;

  try
    if AProcess.Running then begin
                               AProcess.Terminate(0);
                               sleep(100);
                               write(#13#10, '>>> connection closed (', AProcess.ExitCode, ') <<<', #13#10#10)
                             end
                        else write(#13#10, '>>> connection failed (', AProcess.ExitCode, ') <<<', #13#10#10);
{
    sleep(100);
    I:=AProcess.Stderr.NumBytesAvailable;
    if I<>0 then
    begin
      SetLength(buffer,AProcess.Stderr.Read(buffer[1], min(I, 255)));
      buffer:=trim(buffer);
      I:=pos(': ', buffer);

      while I<>0 do
      begin
        buffer[I  ]:=#13;
        buffer[I+1]:=#10;
        I:=pos(': ', buffer)
      end;

      write(buffer, #13#10);
    end;
}
    AProcess.Free
  except write(#10#13, '!! exception 2', #13#10) end;

  TCSetAttr(0, TCSANOW, SAVEDtios)
end.
